﻿#region << Using Directives >>
using System;
#endregion

namespace Volpe.Cafe.Generic
{
    /// <summary>
    /// Represents an object that stores compliance modeling data for each safety class.
    /// </summary>
    [Serializable]
    public class SCValue<T> : ICloneable
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="SCValue{T}"/> class.
        /// </summary>
        public SCValue()
        {
            this.Items = new T[Classes.Length];
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="SCValue{T}"/> class using the specified values.
        /// </summary>
        public SCValue(T pc, T lt, T cm)
        {
            this.Items = new T[] { pc, lt, cm };
        }

        #endregion

        #region /*** Methods ***/

        #region /* Overloaded Operators */

        /// <summary>
        /// Determines whether the specified <see cref="SCValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="SCValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool operator ==(SCValue<T> value1, SCValue<T> value2)
        {
            return Equals(value1, value2);
        }
        /// <summary>
        /// Determines whether the specified <see cref="SCValue{T}"/> values are not equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="SCValue{T}"/> values are not equal; false, otherwise.</returns>
        public static bool operator !=(SCValue<T> value1, SCValue<T> value2)
        {
            return !Equals(value1, value2);
        }

        #endregion

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a shallow copy of the current <see cref="SCValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="SCValue{T}"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a shallow copy of the current <see cref="SCValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="SCValue{T}"/>.</returns>
        public SCValue<T> Clone()
        {
            SCValue<T> value = new SCValue<T>();
            this.CopyTo(value);
            return value;
        }
        /// <summary>
        /// Copies the members of the current <see cref="SCValue{T}"/> instance into the specified value.
        /// </summary>
        /// <param name="value">The value where to copy the members of the current instance.</param>
        protected void CopyTo(SCValue<T> value)
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                value.Items[i] = this.Items[i];
            }
        }

        #endregion

        #region /* Overriden from object */

        /// <summary>
        /// Returns the string representation of this <see cref="SCValue{T}"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="SCValue{T}"/> instance.</returns>
        public override string ToString()
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + this.Items[i].ToString());
            }
            return "{" + s + "}";
        }

        /// <summary>
        /// Serves as a hash function for a particular type, suitable for use in hashing algorithms and data structures like a hash table.
        /// </summary>
        /// <returns>A hash code for the current <see cref="SCValue{T}"/>.</returns>
        public override int GetHashCode()
        {
            int hash = 0;
            for (int i = 0; i < this.Items.Length; i++)
            {
                hash = hash ^ this.Items[i].GetHashCode();
            }
            return hash;
        }

        /// <summary>
        /// Determines whether the specified <see cref="Object"/> is equal to the current <see cref="SCValue{T}"/> value.
        /// </summary>
        /// <param name="obj">The <see cref="Object"/> to compare with the current <see cref="SCValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="Object"/> is equal to the current <see cref="SCValue{T}"/> value; false, otherwise.</returns>
        public override bool Equals(object obj)
        {
            return (obj is SCValue<T>) ? this.Equals((SCValue<T>)obj) : base.Equals(obj);
        }
        /// <summary>
        /// Determines whether the specified <see cref="SCValue{T}"/> value is equal to the current <see cref="SCValue{T}"/> value.
        /// </summary>
        /// <param name="value">The <see cref="SCValue{T}"/> value to compare with the current <see cref="SCValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="SCValue{T}"/> value is equal to the current <see cref="SCValue{T}"/> value;
        ///   false, otherwise.</returns>
        public bool Equals(SCValue<T> value)
        {
            return Equals(this, value);
        }
        /// <summary>
        /// Determines whether the specified <see cref="SCValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first <see cref="SCValue{T}"/> value to compare.</param>
        /// <param name="value2">The second <see cref="SCValue{T}"/> value to compare.</param>
        /// <returns>true, if the specified <see cref="SCValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool Equals(SCValue<T> value1, SCValue<T> value2)
        {
            if (object.ReferenceEquals(value1, null) && object.ReferenceEquals(value2, null)) { return true; }
            if (object.ReferenceEquals(value1, null) || object.ReferenceEquals(value2, null)) { return false; }
            //
            for (int i = 0; i < value1.Items.Length; i++)
            {
                if (!value1.Items[i].Equals(value2.Items[i])) { return false; }
            }
            return true;
        }

        #endregion

        /// <summary>
        /// Resets all members of this <see cref="SCValue{T}"/> instance to their default values (0, false, or null).
        /// </summary>
        public void Clear()
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                this.Items[i] = default(T);
            }
        }

        /// <summary>
        /// Returns the index corresponding to the specified <see cref="SafetyClass"/>.
        /// </summary>
        /// <param name="safetyClass"></param>
        /// <returns>The index corresponding to the specified <see cref="SafetyClass"/>.</returns>
        protected int GetIndex(SafetyClass safetyClass)
        {
            return (int)safetyClass - 1;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets or sets the compliance modeling data value based on the given safety class.</summary>
        /// <exception cref="System.IndexOutOfRangeException">The value specified does not represent a valid safety class.</exception>
        public T this[SafetyClass safetyClass]
        {
            get { return this.Items[this.GetIndex(safetyClass)]; }
            set { this.Items[this.GetIndex(safetyClass)] = value; }
        }

        /// <summary>Gets an array of compliance modeling data specified by <see cref="SafetyClass"/>.</summary>
        public T[] Items { get; private set; }

        ///// <summary>Gets or sets the compliance modeling data for "PC" safety class.</summary>
        //public T PC { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for "LT/SUV" safety class.</summary>
        //public T LT { get; set; }
        ///// <summary>Gets or sets the compliance modeling data for "CUV/Minivan" safety class.</summary>
        //public T CM { get; set; }

        #endregion

        #region /*** Variables ***/

        /// <summary>Specifies the <see cref="SafetyClass"/>es supported by this object. This field is read-only.</summary>
        public static readonly SafetyClass[] Classes = new SafetyClass[] { SafetyClass.PC,
                                                                           SafetyClass.LT,
                                                                           SafetyClass.CM };
        /// <summary>Provides the friendly name for each of the supported <see cref="SafetyClass"/>es. This field is read-only.</summary>
        public static readonly string[] Names = new string[] { "PC",
                                                               "LT",
                                                               "CM" };

        #endregion

    }
}
